using System;
using System.Diagnostics;
using System.Collections;
using System.Collections.Specialized;
using System.Runtime.Remoting.Messaging;
using System.Runtime.Remoting.Lifetime;

namespace MarcObjects
{
	public class PlayerListEntry
	{
		public string Name;
		public PlayerClient Player;
	};

	/// <summary>
	/// Summary description for ServiceCore.
	/// </summary>
	public class ServiceCore : MarshalByRefObject
	{
		// This is the array of players that are available
		protected ArrayList Players;
		protected MusicLibrary MusicLib;
		string LastExceptionMessage;
		public EventLog EventLog;

		// This is a number that gets bigger every time Ping is called
		int PingSequence = 0;
		
		public ServiceCore()
		{
			MusicLib = new MusicLibrary();
			MusicLib.Open();

			Players = new System.Collections.ArrayList();
		}

		public virtual void LogException(Exception ex)
		{
			// Write it to the event log if we're doing that
			if (EventLog != null)
				EventLog.WriteEntry("Exception caught; message is " + ex.Message);

			LastExceptionMessage = ex.Message;
		}

		public MusicLibrary GetMusicLibrary()
		{
			return MusicLib;
		}

		// The PlayerName is passed in so we don't have to call back
		// to the client and request it from the Player object
		public void RegisterPlayer(string PlayerName, ref PlayerClient Player)
		{
			// Set a long lease on the client object so we don't release it
			ILease lease = (ILease)Player.GetLifetimeService();
			if (lease != null)
			{
				lease.Renew(System.TimeSpan.FromMinutes(60));
			}

			bool NewPlayer = true;

			// Remove it if it's already in the list
			foreach (PlayerListEntry entry in Players)
			{
				if (entry.Name == PlayerName)
				{
					NewPlayer = false;
					Players.Remove(entry);
					break;
				};
			};

			try
			{
				PlayerListEntry entry = new PlayerListEntry();
				entry.Player = Player;
				entry.Name = Player.Name;
				Players.Add(entry);
	
				if (NewPlayer)
					NotifyAllPlayers("PlayerListChanged", null);
			} 
			catch (Exception ex)
			{
				LogException(ex);
			};
		}

		public void NotifyAllPlayers(string Message, string Args)
		{
			foreach (PlayerListEntry entry in Players)
			{
				// Send the notification
				entry.Player.Control(Message, Args);
			};
		}

		public void UnregisterPlayer(string Name)
		{
			foreach (PlayerListEntry entry in Players)
			{
				if (entry.Name == Name)
				{
					Players.Remove(entry);
					return;
				};
			};
		}

		public override Object InitializeLifetimeService()
		{
			return null;
		}

		public StringCollection GetPlayersList()
		{
			StringCollection coll = new StringCollection();
			foreach (PlayerListEntry entry in Players)
			{
				coll.Add(entry.Name);
			};

			return coll;
		}

		public int Ping()
		{
			return PingSequence++;
		}

		[OneWayAttribute]
		public void PlaySong(string FullPath, string PlayerName)
		{
			foreach (PlayerListEntry entry in Players)
			{
				if (entry.Name == PlayerName)
				{
					// Play the song; this shouldn't fail
					// since it's a one way call
					entry.Player.PlaySong(FullPath);
				};
			} 
		}

		[OneWayAttribute]
		public void PlayerControl(string PlayerName, string Command)
		{
			foreach (PlayerListEntry entry in Players)
			{
				if (entry.Name == PlayerName)
				{
					// Send the control message; this shouldn't fail
					// since it's a one way call
					entry.Player.Control(Command, null);
				};
			} 
		}

		public void PingPlayers()
		{
			ArrayList RemoveList = new ArrayList();
			ArrayList PingList = new ArrayList();

			// Copy the players list
			foreach (PlayerListEntry entry in Players)
			{
				PingList.Add(entry);
			};

			// Sweep through pinging each one
			foreach (PlayerListEntry entry in PingList)
			{
				try
				{
					entry.Player.Ping();

					// Renew our lease on the object?
					ILease lease = (ILease)entry.Player.GetLifetimeService();
					if (lease != null)
					{
						lease.Renew(System.TimeSpan.FromMinutes(60));
					}
				} 
				catch (Exception)
				{
					RemoveList.Add(entry);
				};
			};

			// Now remove ones that were invalid
			foreach (PlayerListEntry entry in RemoveList)
			{
				Players.Remove(entry);
			};

			if (RemoveList.Count > 0)
			{
				NotifyAllPlayers("PlayerListChanged", null);
			};
		}
	}
}
