using System;
using System.Collections;
using System.Collections.Specialized;
using System.Data.SqlClient;
using System.Text;

namespace MarcObjects
{
	/// <summary>
	/// A FilterSet is an object that defines criteria for 
	/// selecting songs from the database.  It translates
	/// higher level queries of various sorts into 
	/// SQL and parameter sets for database queries.
	/// </summary>
	[Serializable]
	public class FilterSet
	{
		public FilterSet()
		{
			IncludeOwners = new StringCollection();
			IncludeRatings = new StringCollection();
			IncludeTags = new StringCollection();

			ExcludeOwners = new StringCollection();
			ExcludeRatings = new StringCollection();
			ExcludeTags = new StringCollection();
		}

		public FilterSet(string DirectSql)
		{
			this.DirectSql = DirectSql;
		}

		// Either we use the normal filters, or we have
		// SQL text that we use; if this string is not null,
		// then this is what we use
		string DirectSql = null;

		// If this is not empty, then it's a string to search for
		string FullTextSearch = "";

		// Inclusion filters
		public StringCollection IncludeOwners = null;
		public StringCollection IncludeRatings = null;
		public StringCollection IncludeTags = null;

		// Exclusion filters
		public StringCollection ExcludeOwners = null;
		public StringCollection ExcludeRatings = null;
		public StringCollection ExcludeTags = null;

		int VarCount;

		string AddConditions(SqlCommand cmd, string Condition, StringCollection coll)
		{
			StringBuilder sb = new StringBuilder();

			foreach (string str in coll)
			{
				if (sb.Length > 0)
				{
					sb.Append(" or ");
				};

				string VarName = String.Format("@FilterVar{0}", VarCount++);

				sb.AppendFormat("{0}{1}", Condition, VarName);
				cmd.Parameters.Add(VarName, str);
			};

			if (sb.Length > 0)
			{
				return "(" + sb.ToString() + ")";
			} 
			else
			{
				return "";
			};
		}

		public void SetFullTextSearch(string FullTextSearch)
		{
			this.FullTextSearch = FullTextSearch;
		}

		// Get the SQL where clause that this represents
		public string GetWhereClause(SqlCommand cmd)
		{
			StringBuilder sb = new StringBuilder();

			if (DirectSql != null)
			{
				sb.Append(DirectSql);
			} 
			else
			{
				VarCount = 1;

				string IncludeOwnersCondition = AddConditions(cmd, "user_Name=", IncludeOwners);
				string IncludeTagsCondition = AddConditions(cmd, "tag_Name=", IncludeTags);
				string IncludeRatingsCondition = AddConditions(cmd, "rating_Value=", IncludeRatings);

				string ExcludeOwnersCondition = AddConditions(cmd, "user_Name!=", ExcludeOwners);
				string ExcludeTagsCondition = AddConditions(cmd, "tag_Name!=", ExcludeTags);
				string ExcludeRatingsCondition = AddConditions(cmd, "rating_Value=", ExcludeRatings);

				if (IncludeOwnersCondition.Length > 0)
				{
					sb.Append("(");
					sb.Append(IncludeOwnersCondition);
					sb.Append(")");
				};

				if (IncludeTagsCondition.Length > 0)
				{
					if (sb.Length > 0)
					{
						sb.Append(" And ");
					};

					sb.Append("(");
					sb.Append(IncludeTagsCondition);
					sb.Append(")");
				};

				if (IncludeRatingsCondition.Length > 0)
				{
					if (sb.Length > 0)
					{
						sb.Append(" And ");
					};

					sb.Append("(");
					sb.Append(IncludeRatingsCondition);
					sb.Append(")");
				};

				if (FullTextSearch != null && FullTextSearch.Length > 0)
				{
					if (sb.Length > 0)
					{
						sb.Append(" And ");
					};

					sb.Append("UPPER(song_FullPath) like @FullTextSearch");
					cmd.Parameters.Add("@FullTextSearch", "%" + FullTextSearch.ToUpper() + "%");
				}

				// Remove missing songs
				if (sb.Length > 0)
				{
					sb.Append(" And ");
				};

				sb.Append("song_Missing = 0");
			};

			return sb.ToString();
		}
	}
}
